# scripts/accept_2b.py
"""
Tiny checker for 2b results. Reads a summary CSV and prints PASS/FAIL per row.

Gates (AND):
- s_phi in [-1.15, -0.85]
- s_grad in [-2.2, -1.8]
- max_lambda R^2(alpha ~ 1/b) >= 0.90
- coverage > 0%
"""

import argparse
import math
import pandas as pd


def max_r2(row, lambdas=(0.2, 0.5, 1.0)):
    vals = []
    for l in lambdas:
        r2 = row.get(f"alpha_r2_lambda{l}", float("nan"))
        try:
            vals.append(float(r2))
        except Exception:
            pass
    return max([v for v in vals if math.isfinite(v)], default=float("nan"))


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--summary", required=True, help="results_cc\\summary.csv")
    args = ap.parse_args()

    df = pd.read_csv(args.summary)
    if df.empty:
        print("No rows.")
        return

    def verdict(row):
        sphi_ok  = -1.15 <= row.get("s_phi", 9)  <= -0.85
        sgrad_ok = -2.20 <= row.get("s_grad", 9) <= -1.80
        r2_ok    = max_r2(row) >= 0.90
        cov_ok   = float(row.get("cc_coverage_pct", 0.0)) > 0.0
        ok = all([sphi_ok, sgrad_ok, r2_ok, cov_ok])
        reason = []
        if not sphi_ok:  reason.append(f"s_phi={row.get('s_phi')}")
        if not sgrad_ok: reason.append(f"s_grad={row.get('s_grad')}")
        if not r2_ok:    reason.append(f"maxR2={max_r2(row)}")
        if not cov_ok:   reason.append(f"cov={row.get('cc_coverage_pct')}")
        return ("PASS" if ok else "FAIL"), "; ".join(reason)

    cols = ["gauge","L","b","kappa","f","seed","cc_coverage_pct","s_phi","s_grad"]
    cols = [c for c in cols if c in df.columns]
    for _, row in df.iterrows():
        v, why = verdict(row)
        left = " ".join([f"{c}={row[c]}" for c in cols])
        print(f"{left}  ->  {v}  {('['+why+']') if why else ''}")


if __name__ == "__main__":
    main()
